// **************************************************************************************
// el Comandante                                                    guerilla.bplaced.net
// --------------------------------------------------------------------------------------
// Information:
// VSPI sd-card interface, Custom VSPI Pins, flash Winbond W25Q32FVSIG chip (4 MB)
// Read datas from a file on sd-card and flash datas to Winbond 25Q32FVSIG chip (4 MB)
// Written for the esp32 device, should work with any other mcu with SPI- Interface
// --------------------------------------------------------------------------------------
// Details:
// - Keep care to provide 3V3 and NOT 5V
// - Copy the file (named: flashdat.bin) to a FAT32 formatted sd-card
// - Connect a sd-card breakout modul to the esp32 device
// - Insert the sd-card with the file
// - Connect the Winbond W25Q32FVSIG chip (4 MB) to the esp32 device using the adapter
// - Start the Arduino IDE and load this ino file
// - Upload the file to the esp32 device
// - Be patient, flashing and verifying could take a few minutes
// **************************************************************************************

#include <SPI.h>
#include <SD.h>

// ######################################################################################
// Change values in this area to match your envoirement and desires
// --------------------------------------------------------------------------------------
// Winbond DataFlash commands
#define PAGEPROG      0x02
#define READSTATUS    0x05
#define READDATA      0x03
#define WRITEENABLE   0x06
#define CHIPERASE     0x60
#define READID        0x90 

// Custom VSPI Pins for the SD-Card breakout board and the flash ram - must match the wired hardware
#define sck   27
#define miso  32
#define mosi  33
#define cs    2                                           // chip select sd-card breakout board

// Custom Pins for the flash chip - must match the wired hardware
#define sckF  21
#define misoF 22
#define mosiF 23
#define csF   4                                           // chip select flash ram chip
// ######################################################################################

// Vars
uint32_t  errors;
uint8_t   errorSDCard;
uint8_t   errorFlash;
uint8_t   filedata;
uint8_t   flashdata;
uint32_t  filesize;
uint32_t  i;
uint32_t  vspiBusFrequency  = 2000000;                    // >= 4000000 sd-card mount could fail
File      file;                                           // file object
String    szFilepath = "/flashdat.bin";               	  // file to load

// Create a new SPI class on HSPI or VSPI 
SPIClass vspi = SPIClass(VSPI);


// ======================================================================================
// Simple DataFlash - see http://www.technoblogy.com/show?2JMU
//
// David Johnson-Davies - www.technoblogy.com - 17th July 2019
//
// CC BY 4.0 Licensed under a Creative Commons Attribution 4.0 International license: 
// http://creativecommons.org/licenses/by/4.0/
// --------------------------------------------------------------------------------------
// Adapter circuit modified (without diodes):
// https://onecircuit.blogspot.com/2021/06/0000-0000-0110-0111.html
// --------------------------------------------------------------------------------------

class DF {
  public:
    boolean Setup();
    void BeginRead(unsigned long addr);
    void BeginWrite(void);
    uint8_t ReadByte(void);
    void WriteByte(uint8_t data);
    void EndRead(void);
    void EndWrite(void);
  private:
    unsigned long addr;
    uint8_t Read(void);
    void Write(uint8_t);
    void Busy(void);
    void WriteEnable(void);
};

boolean DF::Setup () {
  uint8_t manID, devID;
  //pinMode(power, OUTPUT); digitalWrite(power, HIGH);
  digitalWrite(csF, HIGH); pinMode(csF, OUTPUT); 
  pinMode(sckF, OUTPUT);
  pinMode(mosiF, OUTPUT);
  pinMode(misoF, INPUT);
  digitalWrite(sckF, LOW); digitalWrite(mosiF, HIGH);
  delay(1);
  digitalWrite(csF, LOW);
  delay(100);
  Write(READID); Write(0);Write(0);Write(0);
  manID = Read();
  devID = Read();
  digitalWrite(csF, HIGH);
  return (devID == 0x15);                                 // Found correct device
}

void DF::Write (uint8_t data) {
  shiftOut(mosiF, sckF, MSBFIRST, data);
}

void DF::Busy () {
  digitalWrite(csF, 0);
  Write(READSTATUS);
  while (Read() & 1 != 0);
  digitalWrite(csF, 1);
}

void DF::WriteEnable () {
  digitalWrite(csF, 0);
  Write(WRITEENABLE);
  digitalWrite(csF, 1);
}

void DF::BeginRead (unsigned long start) {
  addr = start;
  Busy();
  digitalWrite(csF, 0);
  Write(READDATA);
  Write(addr>>16);
  Write(addr>>8);
  Write(addr);
}

uint8_t DF::Read () {
  return shiftIn(misoF, sckF, MSBFIRST);
}

void DF::EndRead(void) {
  digitalWrite(csF, 1);
}

void DF::BeginWrite () {
  addr = 0;
  Busy();
  // Erase DataFlash
  WriteEnable();
  digitalWrite(csF, 0);
  Write(CHIPERASE);
  digitalWrite(csF, 1);
  Busy();
}

uint8_t DF::ReadByte () {
  return Read();
}

void DF::WriteByte (uint8_t data) {
  // New page
  if ((addr & 0xFF) == 0) {
    digitalWrite(csF, 1);
    Busy();
    WriteEnable();
    digitalWrite(csF, 0);
    Write(PAGEPROG);
    Write(addr>>16);
    Write(addr>>8);
    Write(0);
  }
  Write(data);
  addr++;
}

void DF::EndWrite (void) {
  digitalWrite(csF, 1);
}

DF DataFlash; 



// Setup
void setup()
{
  // serial monitor - init
  Serial.begin(115200);
  //while (!Serial);
  delay (2000);
  Serial.println("-----------------------------------------------------------");
  Serial.println("Testing");
  Serial.println("-----------------------------------------------------------");
  Serial.println("Serial Monitor: Okay");

  Serial.println("-----------------------------------------------------------");
  Serial.println("Test SD-Card");
  Serial.println("-----------------------------------------------------------");
  // sd-card - vspi init
  digitalWrite(cs, HIGH);                                 // slave select pin csSD (sd-card)
  vspi.begin(sck, miso, mosi, cs);                        // init sd-card spi communication protocol
  Serial.println("SD-Card VSPI: Okay");
  
  // sd-card - mount
  errorSDCard = 0;
  if (!SD.begin(cs, vspi, vspiBusFrequency))
  {Serial.println("SD-Card Mount: Error"); errorSDCard++;}
  else
  {Serial.println("SD-Card Mount: Okay");}
  
  // sd-card - check
  if(SD.cardType() == CARD_NONE)
  {Serial.println("SD-Card Type: None"); errorSDCard++;}
  else
  {Serial.println("SD-Card Type: Okay");}

  // sd-card - open file and read filesize
  file = SD.open(szFilepath, FILE_READ);
  if (file)
  {Serial.print("File Open: "); Serial.println(szFilepath); filesize = file.size(); Serial.print("File Size: "); Serial.println(filesize);}
  else
  {Serial.print("File Open: Error "); errorSDCard++;}

  filedata = file.read();
  Serial.print("File Byte 0: "); Serial.println(filedata, HEX);
  filedata = file.read();
  Serial.print("File Byte 1: "); Serial.println(filedata, HEX);
  file.seek(filesize);
  filedata = file.read();
  Serial.print("File Byte "); Serial.print(filesize, DEC); Serial.print(": "); Serial.println(filedata, HEX);
  file.seek(0);
  filedata = file.read();
  Serial.print("File Byte 0: "); Serial.println(filedata, HEX);

  Serial.println("-----------------------------------------------------------");
  Serial.println("Test Flash");
  Serial.println("-----------------------------------------------------------");
  // flash ram - setup
  errorFlash = 0;
  if (!DataFlash.Setup())
  {Serial.println("Flash chip: Not found"); errorFlash++;}
  else
  {Serial.println("Flash chip: Found");}

  // flash write test
  flashdata = 0x4B;
  Serial.print("Flash Write Test: "); Serial.println(flashdata, HEX);
  DataFlash.BeginWrite();
  DataFlash.WriteByte(flashdata);
  DataFlash.EndWrite();

  // flash read test
  DataFlash.BeginRead(0);
  flashdata = 0;
  flashdata = DataFlash.ReadByte();
  Serial.print("Flash Read Test: "); Serial.println(flashdata, HEX);
  DataFlash.EndRead();

  Serial.println("-----------------------------------------------------------");
  Serial.println("Test Result");
  Serial.println("-----------------------------------------------------------");
  if (errorSDCard == 0 && errorFlash == 0)
  {Serial.println("Okay");}
  else
  {Serial.println("Error"); return;}

  Serial.println("");

  Serial.println("-----------------------------------------------------------");
  Serial.println("Flashing...");
  Serial.println("-----------------------------------------------------------");
  delay (1000);

  // file pointer
  file.seek(0);

  // flash write begin
  DataFlash.BeginWrite();

  // flash write loop
  for (i = 0; i < filesize; i++)
  {
  filedata = file.read();                                   // read byte from file
  DataFlash.WriteByte(filedata);                            // write byte to flash
  }

  // flash write end
  DataFlash.EndWrite();

  // empty line
  Serial.println("");

  Serial.println("-----------------------------------------------------------");
  Serial.println("Verifying...");
  Serial.println("-----------------------------------------------------------");
  delay (1000);

  // file pointer
  file.seek(0);

  // flash read begin
  DataFlash.BeginRead(0);

  // flash read loop
  errors = 0;
  for (i = 0; i < filesize; i++)
  {
    filedata = file.read();
    flashdata = DataFlash.ReadByte();
    //Serial.print("Verify: "); Serial.print("File = "); Serial.print(filedata, HEX); Serial.print(" --- Flash = "); Serial.println(flashdata, HEX);
    if (flashdata != filedata)
    {errors++;}
  }

  // flash read end
  DataFlash.EndRead();

  // empty line
  Serial.println("");

  Serial.println("-----------------------------------------------------------");
  Serial.println("End of program");
  Serial.println("-----------------------------------------------------------");
  file.close();
  Serial.print("Errors: "); Serial.println(errors);
}



// Loop
void loop()
{
// notting to do
}
